document.addEventListener("DOMContentLoaded", function () {
  const progressItems = document.querySelectorAll(".progress-item");
  const sections = document.querySelectorAll(".section");
  const continueButtons = document.querySelectorAll(".btn-continue");
  const backButtons = document.querySelectorAll(".btn-back");
  const submitButton = document.getElementById("submitForm");

  // File upload variables
  const fileInput = document.getElementById("fileInput");
  const fileUploadContainer = document.getElementById("fileUploadContainer");
  const filePreviews = document.getElementById("filePreviews");
  const uploadStatus = document.getElementById("uploadStatus");

  let currentSection = 1;
  let uploadedFiles = [];

  // Update progress tracker
  function updateProgressTracker() {
    // Update progress items
    progressItems.forEach((item) => {
      const itemSection = parseInt(item.getAttribute("data-section"));

      if (itemSection < currentSection) {
        item.classList.add("completed");
        // Keep the check-circle icon but change color to green
        item.querySelector(".progress-icon").innerHTML =
          '<i class="ph ph-check-circle" style="color: #4ade80;"></i>';
      } else if (itemSection === currentSection) {
        item.classList.add("active");
        item.classList.remove("completed");
        // Keep the check-circle icon but change color to white
        item.querySelector(".progress-icon").innerHTML =
          '<i class="ph ph-check-circle" style="color: white;"></i>';
      } else {
        item.classList.remove("active", "completed");
        // Keep the check-circle icon with default color
        item.querySelector(".progress-icon").innerHTML =
          '<i class="ph ph-check-circle"></i>';
      }
    });
  }

  // Show specific section
  function showSection(sectionNumber) {
    sections.forEach((section) => {
      section.classList.remove("active");
    });

    const targetSection = document.getElementById(`section${sectionNumber}`);
    if (targetSection) {
      targetSection.classList.add("active");
      currentSection = sectionNumber;
      updateProgressTracker();
    }
  }

  // Continue button event listeners
  continueButtons.forEach((button) => {
    button.addEventListener("click", function () {
      const nextSection = parseInt(this.getAttribute("data-next"));
      if (nextSection) {
        showSection(nextSection);
      }
    });
  });

  // Back button event listeners
  backButtons.forEach((button) => {
    button.addEventListener("click", function () {
      const prevSection = parseInt(this.getAttribute("data-prev"));
      if (prevSection) {
        showSection(prevSection);
      }
    });
  });

  // Progress item click event listeners
  progressItems.forEach((item) => {
    item.addEventListener("click", function () {
      const sectionNumber = parseInt(this.getAttribute("data-section"));
      showSection(sectionNumber);
    });
  });

  // File upload functionality
  function setupFileUpload() {
    // Handle file selection
    fileInput.addEventListener("change", handleFileSelect);

    // Drag and drop functionality
    fileUploadContainer.addEventListener("dragover", function (e) {
      e.preventDefault();
      fileUploadContainer.classList.add("dragover");
    });

    fileUploadContainer.addEventListener("dragleave", function () {
      fileUploadContainer.classList.remove("dragover");
    });

    fileUploadContainer.addEventListener("drop", function (e) {
      e.preventDefault();
      fileUploadContainer.classList.remove("dragover");

      if (e.dataTransfer.files.length) {
        handleFiles(e.dataTransfer.files);
      }
    });
  }

  function handleFileSelect(e) {
    if (e.target.files.length) {
      handleFiles(e.target.files);
    }
  }

  function handleFiles(files) {
    for (let i = 0; i < files.length; i++) {
      const file = files[i];

      // Check file size (5MB limit)
      if (file.size > 5 * 1024 * 1024) {
        showUploadStatus(
          `File "${file.name}" is too large. Maximum size is 5MB.`,
          "error"
        );
        continue;
      }

      // Check file type
      const validTypes = [
        "image/jpeg",
        "image/jpg",
        "image/png",
        "application/pdf",
      ];
      if (!validTypes.includes(file.type)) {
        showUploadStatus(
          `File "${file.name}" is not a supported format.`,
          "error"
        );
        continue;
      }

      // Add file to uploaded files
      uploadedFiles.push(file);

      // Create preview
      createFilePreview(file);

      // Show success message
      showUploadStatus(`File "${file.name}" uploaded successfully.`, "success");
    }

    // Reset file input
    fileInput.value = "";
  }

  function createFilePreview(file) {
    const filePreview = document.createElement("div");
    filePreview.className = "file-preview";
    filePreview.dataset.fileName = file.name;

    const fileInfo = document.createElement("div");
    fileInfo.className = "file-info";

    const fileName = document.createElement("div");
    fileName.className = "file-name";
    fileName.textContent = file.name;

    const fileSize = document.createElement("div");
    fileSize.className = "file-size";
    fileSize.textContent = formatFileSize(file.size);

    const fileActions = document.createElement("div");
    fileActions.className = "file-actions";

    const deleteBtn = document.createElement("button");
    deleteBtn.className = "file-action-btn";
    deleteBtn.innerHTML = '<i class="ph ph-trash"></i>';
    deleteBtn.addEventListener("click", function () {
      deleteFile(file.name);
      filePreview.remove();
    });

    fileActions.appendChild(deleteBtn);
    fileInfo.appendChild(fileName);
    fileInfo.appendChild(fileSize);

    // Create preview based on file type
    if (file.type.startsWith("image/")) {
      const reader = new FileReader();
      reader.onload = function (e) {
        const img = document.createElement("img");
        img.src = e.target.result;
        filePreview.insertBefore(img, fileInfo);
      };
      reader.readAsDataURL(file);
    } else {
      // For PDF files, show a document icon
      const docIcon = document.createElement("div");
      docIcon.style.textAlign = "center";
      docIcon.style.marginBottom = "10px";
      docIcon.innerHTML =
        '<i class="ph ph-file-text" style="font-size: 48px; color: #ef4444;"></i>';
      filePreview.insertBefore(docIcon, fileInfo);
    }

    filePreview.appendChild(fileInfo);
    filePreview.appendChild(fileActions);
    filePreviews.appendChild(filePreview);
  }

  function deleteFile(fileName) {
    uploadedFiles = uploadedFiles.filter((file) => file.name !== fileName);
    showUploadStatus(`File "${fileName}" deleted.`, "success");
  }

  function formatFileSize(bytes) {
    if (bytes === 0) return "0 Bytes";
    const k = 1024;
    const sizes = ["Bytes", "KB", "MB", "GB"];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
  }

  function showUploadStatus(message, type) {
    uploadStatus.textContent = message;
    uploadStatus.className = `upload-status ${type}`;
    uploadStatus.style.display = "block";

    // Hide status after 3 seconds
    setTimeout(() => {
      uploadStatus.style.display = "none";
    }, 3000);
  }

  // Submit button event listener
  if (submitButton) {
    submitButton.addEventListener("click", function () {
      alert(
        "Form submitted successfully! Thank you for completing the onboarding process."
      );
      // In a real application, you would submit the form data here
    });
  }

  // Initialize progress tracker
  updateProgressTracker();

  // Initialize file upload
  setupFileUpload();
});
